import { json } from '@sveltejs/kit';
import { db } from '$lib/server/db.js';

/**
 * Resolve id (bisa student.id atau user.id) ke baris siswa di tabel students.
 * Panel admin kadang mengirim user.id karena data di-merge dari users + students.
 */
async function resolveStudentById(id) {
  const [byStudentId] = await db.query('SELECT * FROM students WHERE id = ?', [id]);
  if (byStudentId.length > 0) return { student: byStudentId[0], studentId: byStudentId[0].id };

  const [userRows] = await db.query(
    'SELECT id, email FROM users WHERE id = ? AND role = ?',
    [id, 'siswa']
  );
  if (userRows.length === 0) return null;
  const userEmail = userRows[0].email;

  const [byEmail] = await db.query('SELECT * FROM students WHERE email = ?', [userEmail]);
  if (byEmail.length > 0) return { student: byEmail[0], studentId: byEmail[0].id };
  return null;
}

/** @type {import('./$types').RequestHandler} */
export async function PUT({ params, request }) {
  const { id } = params;

  try {
    const body = await request.json();
    const {
      nis,
      fullName,
      email,
      phone,
      address,
      class: className,
      major,
      gender,
      religion,
      status = 'aktif'
    } = body;

    if (!nis || !fullName || !email) {
      return json(
        { error: 'NIS, Nama, dan Email wajib diisi' },
        { status: 400 }
      );
    }

    // Resolve id (bisa student.id atau user.id) ke data siswa
    let resolved = await resolveStudentById(id);
    let studentId;

    if (resolved) {
      studentId = resolved.studentId;
    } else {
      // Id mungkin user.id; siswa hanya ada di users, belum ada di students (mis. tambah siswa gagal insert students).
      const [userRows] = await db.query(
        'SELECT id, email, fullName, phone FROM users WHERE id = ? AND role = ?',
        [id, 'siswa']
      );
      if (userRows.length === 0) {
        return json({ error: 'Siswa tidak ditemukan' }, { status: 404 });
      }
      const u = userRows[0];
      // Buat baris di students agar users dan students tetap sinkron
      const finalClassName = className || null;
      const [insertResult] = await db.query(
        `INSERT INTO students (nis, fullName, email, phone, address, \`class\`, major, gender, religion, status, createdAt)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())`,
        [
          nis,
          fullName.trim(),
          email.trim(),
          phone ? phone.trim() : u.phone,
          address || null,
          finalClassName,
          major || null,
          gender || null,
          religion || null,
          status
        ]
      );
      studentId = insertResult.insertId;
      resolved = {
        student: {
          id: studentId,
          email: email.trim(),
          class: finalClassName
        },
        studentId
      };
    }

    const oldStudent = resolved.student;
    const oldClass = oldStudent.class;
    const finalClassName = className || oldClass || null;

    // Update tabel students (pakai studentId yang sudah di-resolve)
    await db.query(
      `UPDATE students
       SET nis = ?, fullName = ?, email = ?, phone = ?, address = ?, \`class\` = ?, major = ?, gender = ?, religion = ?, status = ?
       WHERE id = ?`,
      [
        nis,
        fullName.trim(),
        email.trim(),
        phone ? phone.trim() : null,
        address || null,
        finalClassName,
        major || null,
        gender || null,
        religion || null,
        status,
        studentId
      ]
    );

    // Sinkronkan dengan tabel users (akun login siswa)
    try {
      await db.query(
        `UPDATE users
         SET fullName = ?, email = ?, phone = ?, major = ?, status = ?
         WHERE role = 'siswa' AND email = ?`,
        [
          fullName.trim(),
          email.trim(),
          phone ? phone.trim() : null,
          major || null,
          status,
          oldStudent.email // gunakan email lama sebagai kunci
        ]
      );
    } catch (userSyncError) {
      console.error('❌ Error sinkronisasi users untuk siswa:', userSyncError);
      // Jangan gagalkan request hanya karena sinkronisasi akun gagal
    }

    // Update statistik kelas (classes.studentCount) secara best-effort
    try {
      // Pastikan kelas baru ada & update jumlah siswa
      if (finalClassName) {
        const [existingClasses] = await db.query(
          'SELECT * FROM classes WHERE name = ?',
          [finalClassName]
        );

        const [studentsInNewClass] = await db.query(
          'SELECT COUNT(*) AS count FROM students WHERE class = ? AND status = "aktif"',
          [finalClassName]
        );

        const newCount = studentsInNewClass[0]?.count || 0;

        if (existingClasses.length > 0) {
          await db.query(
            'UPDATE classes SET studentCount = ? WHERE name = ?',
            [newCount, finalClassName]
          );
        } else {
          await db.query(
            `INSERT INTO classes (name, major, homeroomTeacher, studentCount, status, createdAt)
             VALUES (?, ?, '', ?, 'aktif', NOW())`,
            [finalClassName, major || oldStudent.major || '', newCount || 1]
          );
        }
      }

      // Jika kelas berubah, perbarui juga kelas lama
      if (oldClass && oldClass !== finalClassName) {
        const [studentsInOldClass] = await db.query(
          'SELECT COUNT(*) AS count FROM students WHERE class = ? AND status = "aktif"',
          [oldClass]
        );

        const oldCount = studentsInOldClass[0]?.count || 0;

        await db.query(
          'UPDATE classes SET studentCount = ? WHERE name = ?',
          [oldCount, oldClass]
        );
      }
    } catch (classError) {
      console.error('⚠️ Error update statistik kelas setelah edit siswa:', classError);
      // Best-effort saja, jangan gagalkan update siswa
    }

    return json({
      success: true,
      message: 'Data siswa berhasil diperbarui',
      data: {
        id: Number(studentId),
        nis,
        fullName,
        email,
        phone,
        address,
        class: finalClassName,
        major,
        gender,
        religion,
        status
      }
    });
  } catch (error) {
    console.error('❌ Update siswa API error:', error);
    return json(
      { error: 'Terjadi kesalahan pada server saat mengupdate siswa' },
      { status: 500 }
    );
  }
}

/** @type {import('./$types').RequestHandler} */
export async function DELETE({ params, url }) {
  const { id } = params;
  const action = (url.searchParams.get('action') || 'nonaktif').toLowerCase(); // permanent | nonaktif

  try {
    let resolved = await resolveStudentById(id);
    const [userRows] = await db.query(
      'SELECT id, email FROM users WHERE id = ? AND role = ?',
      [id, 'siswa']
    );
    const userOnly = userRows.length > 0 && !resolved;

    if (action === 'nonaktif') {
      // Nonaktifkan sementara: update status di MySQL, akun tidak bisa login
      if (userOnly) {
        await db.query(`UPDATE users SET status = 'nonaktif' WHERE id = ?`, [id]);
        return json({
          success: true,
          message: 'Siswa berhasil dinonaktifkan. Akun tidak dapat login lagi.'
        });
      }
      if (!resolved) {
        return json({ error: 'Siswa tidak ditemukan' }, { status: 404 });
      }
      const student = resolved.student;
      const studentId = resolved.studentId;
      await db.query(`UPDATE students SET status = 'nonaktif' WHERE id = ?`, [studentId]);
      await db.query(
        `UPDATE users SET status = 'nonaktif' WHERE role = 'siswa' AND email = ?`,
        [student.email]
      );
      return json({
        success: true,
        message: 'Siswa berhasil dinonaktifkan. Akun tidak dapat login lagi.'
      });
    }

    // action === 'permanent': hapus dari database, tersinkron dengan MySQL
    if (userOnly) {
      await db.query('DELETE FROM users WHERE id = ? AND role = ?', [id, 'siswa']);
      return json({
        success: true,
        message: 'Data siswa berhasil dihapus permanen dari database.'
      });
    }
    if (!resolved) {
      return json({ error: 'Siswa tidak ditemukan' }, { status: 404 });
    }

    const student = resolved.student;
    const studentId = resolved.studentId;
    const studentClass = student.class;
    const studentEmail = student.email;

    // Hapus data terkait (urutan sesuai FK)
    await db.query('DELETE FROM subjectAssignments WHERE studentId = ?', [studentId]);
    await db.query('DELETE FROM grades WHERE studentId = ?', [studentId]);
    await db.query('DELETE FROM attendance WHERE studentId = ?', [studentId]);
    await db.query('DELETE FROM studentPayments WHERE studentId = ?', [studentId]);
    await db.query('DELETE FROM students WHERE id = ?', [studentId]);
    await db.query('DELETE FROM users WHERE role = ? AND email = ?', ['siswa', studentEmail]);

    // Update statistik kelas
    try {
      if (studentClass) {
        const [rows] = await db.query(
          'SELECT COUNT(*) AS count FROM students WHERE `class` = ? AND status = "aktif"',
          [studentClass]
        );
        const count = rows[0]?.count ?? 0;
        await db.query(
          'UPDATE classes SET studentCount = ? WHERE name = ?',
          [count, studentClass]
        );
      }
    } catch (classErr) {
      console.error('⚠️ Error update statistik kelas:', classErr);
    }

    return json({
      success: true,
      message: 'Data siswa berhasil dihapus permanen dari database.'
    });
  } catch (error) {
    console.error('❌ Hapus siswa API error:', error);
    return json(
      { error: 'Terjadi kesalahan pada server saat menghapus siswa' },
      { status: 500 }
    );
  }
}

